// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_BROWSER_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_BROWSER_ENDPOINT_H

#include "common.h"
#include "ProtocolEndpointBase.h"
#include "util/Mutex.h"
#include "IMediaBrowserCallbacks.h"

/**
 * This class implements an endpoint that allows browsing of media sources over
 * the GAL protocol. You must also subclass IMediaBrowserCallbacks and implement all
 * the methods so that the GAL receiver can interface with your hardware. The call
 * sequence would look something like this:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      MediaBrowserEndpoint* endpoint = new MediaBrowserEndpoint(serviceId,
 *              galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IMediaBrowserCallbacks.
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 * </pre>
 */
class MediaBrowserEndpoint : public ProtocolEndpointBase {
public:
  MediaBrowserEndpoint(uint8_t id, MessageRouter *router) :
            ProtocolEndpointBase(id, router, false) { }

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<IMediaBrowserCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Fetches the given node. Node will be returned via callbacks.
     * @param path Full path to node to fetch
     * @param start If this node is a List or Source node, start listing leaves at this index.
     * @param get_album_art If true, return album art if available.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int mediaGetNode(const string& path, uint32_t start, bool get_album_art);

    /**
     * Reports an action associated with current phone status.
     * @param caller_number The number of the call to act upon. Can be empty.
     * @param caller_id The caller id to act upon. Can be empty.
     * @param action See enum InstrumentClusterAction in protos.proto for possible values.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int reportAction(const string& path, uint32_t action);

private:
    shared_ptr<IMediaBrowserCallbacks> mCallbacks;
    void handleRootNode(const MediaRootNode& node);
    void handleSourceNode(const MediaSourceNode& node);
    void handleListNode(const MediaListNode& node);
    void handleSongNode(const MediaSongNode& node);
    static void copySourceNode(const MediaSource& src, struct MediaSourceStruct* dest);
    static void copyListNode(const MediaList& src, struct MediaListStruct* dest);
    static void copySongNode(const MediaSong& src, struct MediaSongStruct* dest);
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_BROWSER_ENDPOINT_H
